# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Workflow that creates a child workspace from the one it's run on."""

from debusine.server.tasks.models import CreateChildWorkspaceData
from debusine.server.workflows.base import Workflow
from debusine.server.workflows.models import (
    ChildWorkspaceData,
    WorkRequestWorkflowData,
)
from debusine.tasks.models import BaseDynamicTaskData
from debusine.tasks.server import TaskDatabaseInterface


class CreateChildWorkspaceWorkflow(
    Workflow[ChildWorkspaceData, BaseDynamicTaskData]
):
    """
    Workflow that creates a new child workspace.

    This workflow is a simple interface for users to invoke the
    create_child_workspace server task.
    """

    TASK_NAME = "create_child_workspace"

    def populate(self) -> None:
        """Create the server task that creates the new workspace."""
        task_data = CreateChildWorkspaceData(
            prefix=self.data.prefix or self.workspace.name,
            suffix=self.data.suffix,
            public=self.data.public,
            owner_group=self.data.owner_group,
            workflow_template_names=self.data.workflow_template_names,
            expiration_delay=self.data.expiration_delay,
        )
        self.work_request_ensure_child_server(
            task_name="create_child_workspace",
            task_data=task_data,
            workflow_data=WorkRequestWorkflowData(
                display_name="Create child workspace",
                step="create-child-workspace",
            ),
        )

    def build_dynamic_data(
        self, task_database: TaskDatabaseInterface  # noqa: U100
    ) -> BaseDynamicTaskData:
        """
        Compute dynamic data for this workflow.

        :subject: suffix
        :parameter_summary: suffix
        """
        return BaseDynamicTaskData(
            subject=self.data.suffix, parameter_summary=self.data.suffix
        )
