/*
 * Copyright 2021-2025 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.schema;

import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import static java.util.Collections.unmodifiableMap;

/**
 * Utility class for retrieving schema locations.
 *
 * @since 0.2.2
 */
public final class Schemas {

	private static final Map<Namespace, String> SCHEMAS;

	static {
		Map<Namespace, String> schemas = new HashMap<>();
		schemas.put(Namespace.REPORTING_EVENTS_0_1_0, "/org/opentest4j/reporting/schema/events-0.1.0.xsd");
		schemas.put(Namespace.REPORTING_HIERARCHY_0_1_0, "/org/opentest4j/reporting/schema/hierarchy-0.1.0.xsd");
		schemas.put(Namespace.REPORTING_CORE_0_1_0, "/org/opentest4j/reporting/schema/core-0.1.0.xsd");
		schemas.put(Namespace.REPORTING_JAVA_0_1_0, "/org/opentest4j/reporting/schema/java-0.1.0.xsd");
		schemas.put(Namespace.REPORTING_EVENTS_0_2_0, "/org/opentest4j/reporting/schema/events-0.2.0.xsd");
		schemas.put(Namespace.REPORTING_HIERARCHY_0_2_0, "/org/opentest4j/reporting/schema/hierarchy-0.2.0.xsd");
		schemas.put(Namespace.REPORTING_CORE_0_2_0, "/org/opentest4j/reporting/schema/core-0.2.0.xsd");
		schemas.put(Namespace.REPORTING_GIT_0_2_0, "/org/opentest4j/reporting/schema/git-0.2.0.xsd");
		schemas.put(Namespace.REPORTING_JAVA_0_2_0, "/org/opentest4j/reporting/schema/java-0.2.0.xsd");
		SCHEMAS = unmodifiableMap(schemas);
	}

	/**
	 * {@return the schema location for the given built-in namespace}
	 *
	 * @param namespace the namespace to get the schema location for
	 */
	public static Optional<URL> getSchemaLocation(Namespace namespace) {
		return Optional.ofNullable(SCHEMAS.get(namespace)) //
				.map(Schemas.class::getResource);
	}

	private Schemas() {
	}
}
