/*
 * @(#)Pyraminx.c
 *
 * Copyright 1994 - 2025  David A. Bagley, bagleyd AT verizon.net
 *
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the author not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * This program is distributed in the hope that it will be "useful",
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

/* Methods file for Pyraminx */

#include "file.h"
#include "rngs.h"
#include "sound.h"
#include "PyraminxP.h"
#include "Pyraminx2dP.h"
#ifdef HAVE_OPENGL
#include "PyraminxGLP.h"
#endif

#ifdef WINVER
#ifndef LOGPATH
#define LOGPATH "/usr/tmp"
#endif
#ifndef INIFILE
#define INIFILE "C:/ProgramData/wpuzzles/wpyraminx.ini"
#endif

#define SECTION "setup"

static const char *faceColorString[MAX_FACES] =
{
	"0 0 255",
	"255 0 0",
	"255 255 0",
	"0 255 0"
};

static const char faceColorChar[MAX_FACES] =
{'B', 'R', 'Y', 'G'};
#else

#if defined( USE_SOUND ) && defined( USE_NAS )
Display *dsp;
#endif

#ifndef LOGPATH
#ifdef __VMS
#define LOGPATH "SYS$SCRATCH:"
#else
#define LOGPATH "/usr/tmp"
#endif
#endif

/*static void resizePuzzle(PyraminxWidget w);
static void sizePuzzle(PyraminxWidget w);*/
static Boolean setValuesPuzzle(Widget current, Widget request, Widget renew);
static void destroyPuzzle(Widget old);
static void initializePuzzle(Widget request, Widget renew);
/*static void exposePuzzle(Widget renew, XEvent *event, Region region);*/

PyraminxClassRec pyraminxClassRec =
{
	{
		(WidgetClass) & widgetClassRec,		/* superclass */
		(char *) "Pyraminx",	/* class name */
		sizeof (PyraminxRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) initializePuzzle,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		NULL,		/* actions */
		0,		/* num actions */
		NULL,		/* resources */
		0,		/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		(XtWidgetProc) destroyPuzzle,	/* destroy */
		NULL,		/* resize */
		NULL,		/* expose */
		(XtSetValuesFunc) setValuesPuzzle,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		NULL,		/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		NULL,		/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	}
};

WidgetClass pyraminxWidgetClass = (WidgetClass) & pyraminxClassRec;

void
setPuzzle(PyraminxWidget w, int reason)
{
	pyraminxCallbackStruct cb;

	cb.reason = reason;
	cb.style = -1;
	XtCallCallbacks((Widget) w, (char *) XtNselectCallback, &cb);
}

static void
setPuzzleMove(PyraminxWidget w, int reason, int face, int position,
		int direction, int style, int control, int fast)
{
	pyraminxCallbackStruct cb;

	cb.reason = reason;
	cb.face = face;
	cb.position = position;
	cb.direction = direction;
	cb.style = style;
	cb.control = control;
	cb.fast = fast;
	XtCallCallbacks((Widget) w, (char *) XtNselectCallback, &cb);
}
#endif

static void
loadFont(PyraminxWidget w)
{
#ifndef WINVER
	Display *display = XtDisplay(w);
	const char *altfontname = "-*-times-*-r-*-*-*-180-*";
	char buf[512];

	if (w->pyraminx.fontInfo) {
		XUnloadFont(XtDisplay(w), w->pyraminx.fontInfo->fid);
		XFreeFont(XtDisplay(w), w->pyraminx.fontInfo);
	}
	if (w->pyraminx.font && (w->pyraminx.fontInfo =
			XLoadQueryFont(display, w->pyraminx.font)) == NULL) {
#ifdef HAVE_SNPRINTF
		(void) snprintf(buf, 512,
			"Cannot open %s font.\nAttempting %s font as alternate\n",
			w->pyraminx.font, altfontname);
#else
		(void) sprintf(buf,
			"Cannot open %s font.\nAttempting %s font as alternate\n",
			w->pyraminx.font, altfontname);
#endif
		DISPLAY_WARNING(buf);
		if ((w->pyraminx.fontInfo = XLoadQueryFont(display,
				altfontname)) == NULL) {
#ifdef HAVE_SNPRINTF
			(void) snprintf(buf, 512,
				"Cannot open %s alternate font.\nUse the -font option to specify a font to use.\n",
				altfontname);
#else
			(void) sprintf(buf,
				"Cannot open %s alternate font.\nUse the -font option to specify a font to use.\n",
				altfontname);
#endif
			DISPLAY_WARNING(buf);
		}
	}
	if (w->pyraminx.fontInfo) {
		w->pyraminx.letterOffset.x =
			XTextWidth(w->pyraminx.fontInfo, "8", 1) / 2;
		w->pyraminx.letterOffset.y =
			w->pyraminx.fontInfo->max_bounds.ascent / 2;
	} else
#endif
	{
		w->pyraminx.letterOffset.x = 3;
		w->pyraminx.letterOffset.y = 4;
	}
}

#ifndef LOGFILE
#define LOGFILE "pyraminx.log"
#endif

#define NOTDIR(x) ((x==CW)?CCW:CW)

typedef struct _RowNextP3 {
	int viewChanged, face, direction, reverse;
} RowNextP3;
static int directionNextFace[MAX_ORIENT][MAX_FACES] =
{
	{3, 0, 1, 2},
	{3, 3, 1, 1},
	{1, 3, 3, 1},
	{1, 2, 3, 0},
	{2, 2, 0, 0},
	{2, 0, 0, 2}
};
static PyraminxLoc slideNextRowP2[MAX_SIDES][3] =
{
	{
		{2, 0},
		{1, 3},
		{2, 3}
	},
	{
		{2, 0},
		{0, 3},
		{2, 3}
	}
};
static RowNextP3 slideNextRowP3[MAX_SIDES][MAX_ORIENT] =
{
	{
		{TRUE, UP, TR, FALSE},
		{TRUE, UP, TOP, FALSE},
		{FALSE, UP, BOTTOM, TRUE},
		{FALSE, UP, RIGHT, TRUE},
		{TRUE, DOWN, RIGHT, TRUE},
		{TRUE, DOWN, TR, TRUE}
	},
	{
		{FALSE, DOWN, LEFT, TRUE},
		{TRUE, UP, LEFT, TRUE},
		{TRUE, UP, BL, TRUE},
		{TRUE, DOWN, BL, FALSE},
		{TRUE, DOWN, BOTTOM, FALSE},
		{FALSE, DOWN, TOP, TRUE}
	}
};

/* FIXME */
static int rotOrientRowP3[3][MAX_ORIENT] =
/* current orient, rotation */
{
	{2, 1, 2, 1, 2, 1},
	{0, 2, 0, 2, 0, 2},
	{1, 0, 1, 0, 1, 0}
};

static PyraminxStack undo = {NULL, NULL, NULL, 0};
static PyraminxStack redo = {NULL, NULL, NULL, 0};

static void
checkPieces(PyraminxWidget w)
{
	char *buf1 = NULL, *buf2 = NULL;

	if (w->pyraminx.size < MIN_FACETS) {
		intCat(&buf1,
			"Number of triangles on an edge out of bounds, use at least ",
			MIN_FACETS);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULT_FACETS);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->pyraminx.size = DEFAULT_FACETS;
	}
	if (w->pyraminx.mode < PERIOD2 || w->pyraminx.mode > BOTH) {
		intCat(&buf1, "Mode is in error, use 2 for Period2, 3 for Period3, 4 for Both, defaulting to ",
			DEFAULT_MODE);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->pyraminx.mode = DEFAULT_MODE;
	}
}

Boolean
checkSolved(PyraminxWidget w)
{
	int face, position;
	PyraminxLoc test = {0, FALSE};

	for (face = 0; face < MAX_FACES; face++)
		for (position = 0; position < w->pyraminx.sizeSize; position++) {
			if (!position) {
				test.face = w->pyraminx.facetLoc[face][position].face;
				test.rotation = w->pyraminx.facetLoc[face][position].rotation;
			} else if (test.face !=		/* MAX_SIDES * view + face */
					 w->pyraminx.facetLoc[face][position].face ||
					(w->pyraminx.orient && test.rotation !=
					w->pyraminx.facetLoc[face][position].rotation))
				return False;
		}
	return True;
}

#ifdef DEBUG

 void
printFacets(PyraminxWidget w)
{
	int face, position, square;

	for (face = 0; face < 2 * MAX_SIDES; face++) {
		square = 1;
		for (position = 0; position < w->pyraminx.sizeSize; position++) {
			(void) printf("%d %d  ",
				w->pyraminx.facetLoc[face][position].face,
				w->pyraminx.facetLoc[face][position].rotation);
			if (position == square * square - 1) {
				(void) printf("\n");
				++square;
			}
		}
		(void) printf("\n");
	}
	(void) printf("\n");
}

static void
printFace(PyraminxWidget w)
{
	int g, h, side;

	for (g = 0; g < w->pyraminx.size; g++) {
		for (h = 0; h < w->pyraminx.size - g; h++)
			for (side = 0; side < MAX_SIDES; side++)
				if (!side || h < w->pyraminx.size - g - 1)
					(void) printf("%d %d  ",
						w->pyraminx.faceLoc[side][h +
						g * w->pyraminx.size].face,
						w->pyraminx.faceLoc[side][h +
						g * w->pyraminx.size].rotation);
		(void) printf("\n");
	}
	(void) printf("\n");
}

static void
printRow2(PyraminxWidget w, int orient)
{
	int g, side;

	(void) printf("Row %d:\n", orient);
	for (g = 0; g < w->pyraminx.size; g++)
		for (side = 0; side < MAX_SIDES; side++)
			(void) printf("%d %d  ", w->pyraminx.rowLoc[orient][side][g].face,
				w->pyraminx.rowLoc[orient][side][g].rotation);
	(void) printf("\n");
}

static void
printRow3(PyraminxWidget w, int len, int orient)
{
	int g, side;

	(void) printf("Row %d:\n", orient);
	for (g = 0; g <= len; g++)
		for (side = 0; side < MAX_SIDES; side++)
			if (!side || g < len)
				(void) printf("%d %d  ", w->pyraminx.rowLoc[orient][side][g].face,
				w->pyraminx.rowLoc[orient][side][g].rotation);
	(void) printf("\n");
}

#endif

/* This is fast for small i, a -1 is returned for negative i. */
static int
SQRT(int i)
{
	int j = 0;

	while (j * j <= i)
		j++;
	return (j - 1);
}

void
toCRD(PyraminxWidget w, int face, int position, CRD * crd)
{
	int diag, diag2;

	diag = SQRT(position);
	diag2 = diag * diag;
	if ((face & 1) == UP) {
		crd->column = w->pyraminx.size - 1 - (position - diag2) / 2;
		crd->row = w->pyraminx.size - 1 - (diag2 + 2 * diag - position) / 2;
		crd->diagonal = 2 * w->pyraminx.size - 1 - diag;
	} else {
		crd->column = (position - diag2) / 2;
		crd->row = (diag2 + 2 * diag - position) / 2;
		crd->diagonal = diag;
	}
}

int
toPosition(PyraminxWidget w, CRD crd)
{
	int diag;

	if (crd.diagonal < w->pyraminx.size)
		return (crd.diagonal * crd.diagonal + crd.diagonal +
			crd.column - crd.row);
	diag = 2 * w->pyraminx.size - 1 - crd.diagonal;
	return (diag * diag + diag + crd.row - crd.column);
}

static int
positionCRD(int dir, int i, int j, int side)
{
	if (dir == TOP || dir == BOTTOM)
		return (i);
	else if (dir == RIGHT || dir == LEFT)
		return (j);
	else	/* dir == TR || dir == BL */
		return (i + j + side);
}

static int
length(PyraminxWidget w, int face, int dir, int h)
{
	if (dir == TR || dir == BL)
		return ((face == UP) ? 2 * w->pyraminx.size - 1 - h : h);
	else
		return ((face == UP) ? h : w->pyraminx.size - 1 - h);
}

static void
drawTriangle(PyraminxWidget w, int face, int position, int offset)
{
	if (w->pyraminx.dim == 2)
		drawTriangle2D((Pyraminx2DWidget) w, face, position, offset);
}

static void
drawTriangleAll(PyraminxWidget w, int face, int position, int offset)
{
	drawTriangle(w, face, position, offset);
#ifdef HAVE_OPENGL
	if (w->pyraminx.dim == 4)
		drawAllPiecesGL((PyraminxGLWidget) w);
#endif
}

void
drawAllPieces(PyraminxWidget w)
{
	int face, position;

#ifdef HAVE_OPENGL
	if (w->pyraminx.dim == 4) {
		drawAllPiecesGL((PyraminxGLWidget) w);
		return;
	}
#endif
	for (face = 0; face < MAX_FACES; face++)
		for (position = 0; position < w->pyraminx.sizeSize; position++)
			drawTriangle(w, face, position, FALSE);
}

static void
drawFrame(const PyraminxWidget w, Boolean focus)
{
	if (w->pyraminx.dim == 2)
		drawFrame2D((Pyraminx2DWidget) w, focus);
/*#ifdef HAVE_OPENGL
	else if (w->pyraminx.dim == 4)
		drawFrameGL((PyraminxGLWidget) w, focus);
#endif*/
}

static void
moveNoPieces(PyraminxWidget w)
{
	setPuzzle(w, ACTION_ILLEGAL);
}

static void
rotateFace(PyraminxWidget w, int view, int faceOnView, int direction)
{
	int g, h, side, face, position;
	CRD crd;

	/* Read Face */
	for (g = 0; g < w->pyraminx.size; g++)
		for (h = 0; h < w->pyraminx.size - g; h++)
			for (side = 0; side < MAX_SIDES; side++)
				if (g + h + side < w->pyraminx.size) {
					if (faceOnView == DOWN) {
						crd.column = h;
						crd.row = g;
						crd.diagonal = h + g + side;
						face = view * MAX_SIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
						position = toPosition(w, crd);
						w->pyraminx.faceLoc[side][h + g * w->pyraminx.size] =
							w->pyraminx.facetLoc[face][position];
					} else {	/* faceOnView == UP */
						crd.column = w->pyraminx.size - 1 - h;
						crd.row = w->pyraminx.size - 1 - g;
						crd.diagonal = crd.column + crd.row + !side;
						face = view * MAX_SIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
						position = toPosition(w, crd);
						w->pyraminx.faceLoc[side][h + g * w->pyraminx.size] =
							w->pyraminx.facetLoc[face][position];
					}
				}
	/* Write Face */
	if (faceOnView == DOWN) {
		for (g = 0; g < w->pyraminx.size; g++)
			for (h = 0; h < w->pyraminx.size - g; h++)
				for (side = 0; side < MAX_SIDES; side++)
					if (g + h + side < w->pyraminx.size) {
						crd.column = h;
						crd.row = g;
						crd.diagonal = h + g + side;
						face = view * MAX_SIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
						position = toPosition(w, crd);
						if (direction == CCW)
							w->pyraminx.facetLoc[face][position] =
								w->pyraminx.faceLoc[side][w->pyraminx.size - 1 - g - h - side +
							h * w->pyraminx.size];
						else	/* direction == CW */
							w->pyraminx.facetLoc[face][position] =
								w->pyraminx.faceLoc[side][g +
											(w->pyraminx.size - 1 - g - h - side) * w->pyraminx.size];
						w->pyraminx.facetLoc[face][position].rotation =
							(direction == CCW) ?
							(w->pyraminx.facetLoc[face][position].rotation + 2) %
							MAX_ORIENT :
							(w->pyraminx.facetLoc[face][position].rotation + 4) %
							MAX_ORIENT;
						drawTriangle(w, face, position, FALSE);
					}
	} else {		/* faceOnView == UP */
		for (g = w->pyraminx.size - 1; g >= 0; g--)
			for (h = w->pyraminx.size - 1; h >= w->pyraminx.size - 1 - g; h--)
				for (side = 1; side >= 0; side--)
					if (g + h + side >= w->pyraminx.size) {
						crd.column = h;
						crd.row = g;
						crd.diagonal = h + g + side;
						face = view * MAX_SIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
						position = toPosition(w, crd);
						if (direction == CCW)
							w->pyraminx.facetLoc[face][position] =
								w->pyraminx.faceLoc[!side][g + h - w->pyraminx.size + 1 - !side
											+ (w->pyraminx.size - 1 - h) * w->pyraminx.size];
						else	/* (direction == CW) */
							w->pyraminx.facetLoc[face][position] =
								w->pyraminx.faceLoc[!side][w->pyraminx.size - 1 - g +
											(g + h - w->pyraminx.size + 1 - !side) * w->pyraminx.size];
						w->pyraminx.facetLoc[face][position].rotation =
							(direction == CCW) ?
							(w->pyraminx.facetLoc[face][position].rotation + 2) %
							MAX_ORIENT :
							(w->pyraminx.facetLoc[face][position].rotation + 4) %
							MAX_ORIENT;
						drawTriangle(w, face, position, FALSE);
					}
	}
}

static void
readCRD2(PyraminxWidget w, int view, int dir, int h, int orient)
{
	int g, i, j, side, faceOnView, s, face, position, v = view;
	CRD crd;

	if (dir == TOP || dir == BOTTOM) {
		for (g = 0; g < w->pyraminx.size; g++)
			for (side = 0; side < MAX_SIDES; side++) {
				crd.column = h;
				crd.row = g;
				crd.diagonal = h + g + side;
				face = v * MAX_SIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
				position = toPosition(w, crd);
				w->pyraminx.rowLoc[orient][side][g] =
					w->pyraminx.facetLoc[face][position];
			}
	} else if (dir == RIGHT || dir == LEFT) {
		for (g = 0; g < w->pyraminx.size; g++)
			for (side = 0; side < MAX_SIDES; side++) {
				crd.column = g;
				crd.row = h;
				crd.diagonal = h + g + side;
				face = v * MAX_SIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
				position = toPosition(w, crd);
				w->pyraminx.rowLoc[orient][side][g] =
					w->pyraminx.facetLoc[face][position];
			}
	} else {		/* dir == TR || dir == BL */
		faceOnView = (h < w->pyraminx.size) ? DOWN : UP;
		i = (faceOnView == UP) ? w->pyraminx.size - 1 : 0;
		j = h % w->pyraminx.size;
		for (g = 0; g < w->pyraminx.size; g++) {
			for (side = 0; side < MAX_SIDES; side++) {
				s = (((side == UP) && (faceOnView == UP)) ||
					((side == DOWN) && (faceOnView == DOWN)))
					? DOWN : UP;
				crd.column = i;
				crd.row = j;
				crd.diagonal = i + j + s;
				face = v * MAX_SIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
				position = toPosition(w, crd);
				w->pyraminx.rowLoc[orient][side][g] =
					w->pyraminx.facetLoc[face][position];
				if (side == DOWN) {
					if (faceOnView == UP) {
						if (j == w->pyraminx.size - 1)
							v = (v == UP) ? DOWN : UP;
						j = (j + 1) % w->pyraminx.size;
					} else {	/* faceOnView == DOWN */
						if (!j)
							v = (v == UP) ? DOWN : UP;
						j = (j - 1 + w->pyraminx.size) % w->pyraminx.size;
					}
				}
			}
			i = (faceOnView == UP) ? i - 1 : i + 1;
		}
	}
}

static void
readCRD3(PyraminxWidget w, int view, int faceOnView, int dir, int h, int len, int orient)
{
	int g, i, j, side, s, face, position, faceView = faceOnView;
	CRD crd;

	if (dir == TOP || dir == BOTTOM) {
		for (g = 0; g <= len; g++)
			for (side = 0; side < MAX_SIDES; side++)
				if ((side == DOWN) || g < len) {
					crd.column = h;
					crd.row = (faceView == UP) ? w->pyraminx.size - 1 - g : g;
					crd.diagonal = h + crd.row +
						((((side == UP) && (faceView == DOWN)) ||
						((side == DOWN) && (faceView == UP)))
						? UP : DOWN);
					face = view * MAX_SIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
					position = toPosition(w, crd);
					w->pyraminx.rowLoc[orient][side][g] =
						w->pyraminx.facetLoc[face][position];
				}
	} else if (dir == RIGHT || dir == LEFT) {
		for (g = 0; g <= len; g++)
			for (side = 0; side < MAX_SIDES; side++)
				if ((side == DOWN) || g < len) {
					crd.column = (faceView == UP) ? w->pyraminx.size - 1 - g : g;
					crd.row = h;
					crd.diagonal = h + crd.column +
						((((side == UP) && (faceView == DOWN)) ||
						((side == DOWN) && (faceView == UP)))
						? UP : DOWN);
					face = view * MAX_SIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
					position = toPosition(w, crd);
					w->pyraminx.rowLoc[orient][side][g] =
						w->pyraminx.facetLoc[face][position];
				}
	} else {	/* dir == TR || dir == BL */
		faceView = (h < w->pyraminx.size) ? DOWN : UP;
		i = (faceView == UP) ? w->pyraminx.size - 1 : 0;
		j = h % w->pyraminx.size;
		for (g = 0; g <= len; g++) {
			for (side = 0; side < MAX_SIDES; side++) {
				if (side == DOWN || g < len) {
					s = (((side == UP) && (faceView == UP)) ||
						((side == DOWN) && (faceView == DOWN)))
						? DOWN : UP;
					crd.column = i;
					crd.row = j;
					crd.diagonal = i + j + s;
					face = view * MAX_SIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
					position = toPosition(w, crd);
					w->pyraminx.rowLoc[orient][side][g] =
						w->pyraminx.facetLoc[face][position];
					if (side == DOWN)
						j = (faceView == UP) ? j + 1 : j - 1;
				}
			}
			i = (faceView == UP) ? i - 1 : i + 1;
		}
	}
}

static void
rotateCRD2(PyraminxWidget w, int rotate, int orient)
{
	int g, side;

	for (g = 0; g < w->pyraminx.size; g++)
		for (side = 0; side < MAX_SIDES; side++)
			w->pyraminx.rowLoc[orient][side][g].rotation =
				(w->pyraminx.rowLoc[orient][side][g].rotation + rotate) % 3;
}

static void
rotateCRD3(PyraminxWidget w, int faceOnView, int dir, int len, int orient)
{
	int g, side, direction, facetOrient;

	for (g = 0; g <= len; g++)
		for (side = 0; side < MAX_SIDES; side++)
			if (side == DOWN || g < len) {
				direction = (faceOnView == UP) ? (dir + 3) % MAX_ORIENT : dir;
				facetOrient = w->pyraminx.rowLoc[orient][side][g].rotation % 3;
				w->pyraminx.rowLoc[orient][side][g].rotation =
					rotOrientRowP3[facetOrient][direction];
			}
}

static void
reverseCRD2(PyraminxWidget w, int orient)
{
	PyraminxLoc temp;
	int g;

	for (g = 0; g < w->pyraminx.size; g++) {
		temp = w->pyraminx.rowLoc[orient][g & 1][g / 2];
		w->pyraminx.rowLoc[orient][g & 1][g / 2] =
			w->pyraminx.rowLoc[orient][((g & 1) == 1) ? 0 : 1][w->pyraminx.size - 1 - g / 2];
		w->pyraminx.rowLoc[orient][((g & 1) == 1) ? 0 : 1][w->pyraminx.size - 1 - g / 2] =
			temp;
	}
}

static void
reverseCRD3(PyraminxWidget w, int len, int orient)
{
	PyraminxLoc temp;
	int g;

	for (g = 0; g < len; g++) {
		temp = w->pyraminx.rowLoc[orient][g & 1][len - ((g + 1) / 2)];
		w->pyraminx.rowLoc[orient][g & 1][len - ((g + 1) / 2)] =
			w->pyraminx.rowLoc[orient][g & 1][g / 2];
		w->pyraminx.rowLoc[orient][g & 1][g / 2] = temp;
	}
}

static void
writeCRD2(PyraminxWidget w, int view, int dir, int h, int orient)
{
	int g, side, i, j, s, faceOnView, face, position, v = view;
	CRD crd;

	if (dir == TOP || dir == BOTTOM) {
		for (g = 0; g < w->pyraminx.size; g++)
			for (side = 0; side < MAX_SIDES; side++) {
				crd.column = h;
				crd.row = g;
				crd.diagonal = h + g + side;
				face = v * MAX_SIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
				position = toPosition(w, crd);
				w->pyraminx.facetLoc[face][position] =
					w->pyraminx.rowLoc[orient][side][g];
				drawTriangle(w,
					v * MAX_SIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN),
					toPosition(w, crd), FALSE);
			}
	} else if (dir == RIGHT || dir == LEFT) {
		for (g = 0; g < w->pyraminx.size; g++)
			for (side = 0; side < MAX_SIDES; side++) {
				crd.column = g;
				crd.row = h;
				crd.diagonal = h + g + side;
				face = v * MAX_SIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
				position = toPosition(w, crd);
				w->pyraminx.facetLoc[face][position] =
					w->pyraminx.rowLoc[orient][side][g];
				drawTriangle(w, face, position, FALSE);
			}
	} else {	/* dir == TR || dir == BL */
		faceOnView = (h < w->pyraminx.size) ? DOWN : UP;
		i = (faceOnView == UP) ? w->pyraminx.size - 1 : 0;
		j = h % w->pyraminx.size;
		for (g = 0; g < w->pyraminx.size; g++) {
			for (side = 0; side < MAX_SIDES; side++) {
				s = (((side == UP) && (faceOnView == UP)) ||
					((side == DOWN) && (faceOnView == DOWN)))
					? DOWN : UP;
				crd.column = i;
				crd.row = j;
				crd.diagonal = i + j + s;
				face = v * MAX_SIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
				position = toPosition(w, crd);
				w->pyraminx.facetLoc[face][position] =
					w->pyraminx.rowLoc[orient][side][g];
				drawTriangle(w, face, position, FALSE);
				if (side == DOWN) {
					if (faceOnView == UP) {
						if (j == w->pyraminx.size - 1) {
							v = (v == UP) ? DOWN : UP;
							j = 0;
						} else
							++j;
					} else {	/* FACE == DOWN */
						if (j == 0) {
							v = (v == UP) ? DOWN : UP;
							j = w->pyraminx.size - 1;
						} else
							--j;
					}
				}
			}
			if (faceOnView == UP)
				--i;
			else	/* faceOnView == DOWN */
				++i;
		}
	}
}

static void
writeCRD3(PyraminxWidget w, int view, int faceOnView, int dir, int h, int len, int orient)
{
	int g, side, i, j, k, s, face, position, v = view, faceView = faceOnView;
	CRD crd;

	if (dir == TOP || dir == BOTTOM) {
		for (k = 0; k <= len; k++)
			for (side = 0; side < MAX_SIDES; side++)
				if ((side == DOWN) || k < len) {
					g = (faceView == UP) ? w->pyraminx.size - 1 - k : k;
					s = (((side == UP) && (faceView == DOWN)) ||
						((side == DOWN) && (faceView == UP)))
						? UP : DOWN;
					crd.column = h;
					crd.row = g;
					crd.diagonal = h + g + s;
					face = v * MAX_SIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
					position = toPosition(w, crd);
					w->pyraminx.facetLoc[face][position] =
						w->pyraminx.rowLoc[orient][side][k];
					drawTriangle(w, face, position, FALSE);
				}
	} else if (dir == RIGHT || dir == LEFT) {
		for (k = 0; k <= len; k++)
			for (side = 0; side < MAX_SIDES; side++)
				if ((side == DOWN) || k < len) {
					g = (faceView == UP) ? w->pyraminx.size - 1 - k : k;
					s = (((side == UP) && (faceView == DOWN)) ||
						((side == DOWN) && (faceView == UP)))
						? UP : DOWN;
					crd.column = g;
					crd.row = h;
					crd.diagonal = h + g + s;
					face = v * MAX_SIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
					position = toPosition(w, crd);
					w->pyraminx.facetLoc[face][position] =
						w->pyraminx.rowLoc[orient][side][k];
					drawTriangle(w, face, position, FALSE);
				}
	} else {		/* dir == TR || dir == BL */
		faceView = (h < w->pyraminx.size) ? DOWN : UP;
		i = (faceView == UP) ? w->pyraminx.size - 1 : 0;
		j = h % w->pyraminx.size;
		for (k = 0; k <= len; k++) {
			for (side = 0; side < MAX_SIDES; side++)
				if ((side == DOWN) || k < len) {
					s = (((side == UP) && (faceView == UP)) ||
						((side == DOWN) && (faceView == DOWN)))
						? DOWN : UP;
					crd.column = i;
					crd.row = j;
					crd.diagonal = i + j + s;
					face = v * MAX_SIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
					position = toPosition(w, crd);
					w->pyraminx.facetLoc[face][position] =
						w->pyraminx.rowLoc[orient][side][k];
					drawTriangle(w, face, position, FALSE);
					if (side == DOWN) {
						if (faceView == UP) {
							if (j == w->pyraminx.size - 1) {
								v = (v == UP) ? DOWN : UP;
								j = 0;
							} else
								++j;
						} else {	/* FACE == DOWN */
							if (j == 0) {
								v = (v == UP) ? DOWN : UP;
								j = w->pyraminx.size - 1;
							} else
								--j;
						}
					}
				}
			if (faceView == UP)
				--i;
			else	/* faceView == DOWN */
				++i;
		}
	}
}

static void
movePieces(PyraminxWidget w, int face, CRD crd, int direction, int style,
		Boolean control, int fast)
{
	Boolean reverse, bound;
	int view, side;
	int newSide, newView, rotate, h, k, newH, faceOnView, len;
	int newFace, newDirection, l = 0;
	int dir = direction;

#ifdef HAVE_OPENGL
	if (!control && fast != INSTANT && w->pyraminx.dim == 4) {
		movePiecesGL((PyraminxGLWidget) w, face, toPosition(w, crd),
			direction, style, control, fast);
	}
#endif
	view = face / MAX_SIDES;
	side = crd.diagonal - crd.column - crd.row;
	if (style == PERIOD2) {
		/* Period 2 Slide rows */
		h = positionCRD(dir, crd.column, crd.row, side);
		if (w->pyraminx.sticky && (h % 4 == 1 || h % 4 == 2)) {
			bound = True;
			l = 0;
			if (h % 4 == 2)
				h = h - 1;
		} else
			bound = False;
		do {
			readCRD2(w, view, dir, h, 0);
			for (k = 1; k <= 2; k++) {
				rotate = slideNextRowP2[side][dir % 3].rotation;
				if (dir == TR || dir == BL) {
					newView = view;
					newSide = !side;
					newH = 2 * w->pyraminx.size - 1 - h;
					reverse = False;
				} else if (!rotate) {
					newView = !view;
					newSide = side;
					newH = h;
					reverse = False;
				} else {	/* rotate == 3 */
					newView = !view;
					newSide = !side;
					newH = w->pyraminx.size - 1 - h;
					reverse = True;
				}
				if (k != 2)
					readCRD2(w, newView, dir, newH, k);
				rotateCRD2(w, rotate, k - 1);
				if (reverse)
					reverseCRD2(w, k - 1);
				writeCRD2(w, newView, dir, newH, k - 1);
				view = newView;
				h = newH;
				side = newSide;
			}
			l++;
			h++;
		} while (bound && l < 2);
	} else {		/* style == PERIOD3 */
		faceOnView = (crd.diagonal >= w->pyraminx.size) ? UP : DOWN;
		h = positionCRD(dir, crd.column, crd.row, side);
		bound = False;
		if (dir == TR || dir == BL) {
			if (h == w->pyraminx.size)
				rotateFace(w, view, DOWN, (dir == TR) ? CCW : CW);
			else if (h == w->pyraminx.size - 1)
				rotateFace(w, view, UP, (dir == TR) ? CW : CCW);
			else if (w->pyraminx.sticky)
				bound = True;
		} else if (!crd.column && faceOnView == DOWN &&
				(dir == TOP || dir == BOTTOM))
			rotateFace(w, (view == UP) ? DOWN : UP, DOWN,
			 (dir == TOP || dir == LEFT) ? CCW : CW);
		else if (crd.column == w->pyraminx.size - 1 && faceOnView &&
			 (dir == TOP || dir == BOTTOM))
			rotateFace(w, (view == UP) ? DOWN : UP, UP,
				(dir == BOTTOM || dir == RIGHT) ? CCW : CW);
		else if (crd.row == 0 && faceOnView == DOWN &&
			 (dir == RIGHT || dir == LEFT))
			rotateFace(w, (view == UP) ? DOWN : UP, UP,
				(dir == BOTTOM || dir == RIGHT) ? CCW : CW);
		else if (crd.row == w->pyraminx.size - 1 && faceOnView == UP &&
			 (dir == RIGHT || dir == LEFT))
			rotateFace(w, (view == UP) ? DOWN : UP, DOWN,
			 (dir == TOP || dir == LEFT) ? CCW : CW);
		else if (w->pyraminx.sticky)
			bound = True;
		/* Slide rows */
		if (bound) {
			l = 0;
			if (dir == TR || dir == BL)
				h = (faceOnView == UP) ? w->pyraminx.size + 1 : 0;
			else
				h = (faceOnView == UP) ? 0 : 1;
		}
		do {
			len = length(w, faceOnView, dir, h);
			readCRD3(w, view, faceOnView, dir, h, len, 0);
			for (k = 1; k <= 3; k++) {
				newView = (slideNextRowP3[faceOnView][dir].viewChanged)
					? ((view == UP) ? DOWN : UP) : view;
				newFace = slideNextRowP3[faceOnView][dir].face;
				newDirection = slideNextRowP3[faceOnView][dir].direction;
				reverse = slideNextRowP3[faceOnView][dir].reverse;
				newH = w->pyraminx.size - 1 - h;
				if (faceOnView == UP) {
					if (dir == TR || dir == RIGHT)
						newH = 2 * w->pyraminx.size - 1 - h;
					else if (dir == BOTTOM)
						newH = h;
					else if (dir == BL)
						newH = h - w->pyraminx.size;
				} else {
					if (dir == TOP)
						newH = w->pyraminx.size + h;
					else if (dir == TR)
						newH = h;
				}
				if (k != 3)
					readCRD3(w, newView, newFace, newDirection, newH, len, k);
				rotateCRD3(w, faceOnView, dir, len, k - 1);
				if (reverse)
					reverseCRD3(w, len, k - 1);
				writeCRD3(w, newView, newFace, newDirection, newH, len, k - 1);
				view = newView;
				faceOnView = newFace;
				dir = newDirection;
				h = newH;
			}
			h++;
			l++;
		} while (bound && l < w->pyraminx.size - 1);
	}
#ifdef HAVE_OPENGL
	/* if orient, then needs a redraw to get the lines right */
	/*if (!control && (fast == INSTANT || w->pyraminx.orient) && w->pyraminx.dim == 4)*/
	if (!control && fast == INSTANT && w->pyraminx.dim == 4) {
		drawAllPiecesGL((PyraminxGLWidget) w);
	}
#endif
}

static void
moveControlCb(PyraminxWidget w, int face, int direction, int style, int fast)
{
	int i, faceOnView;
	CRD crd;

#ifdef HAVE_OPENGL
	if (fast != INSTANT && w->pyraminx.dim == 4) {
		movePiecesGL((PyraminxGLWidget) w, face, 0, direction,
			style, True, fast);
	}
#endif
	faceOnView = face % MAX_SIDES;
	if (w->pyraminx.sticky) {
		if (style == PERIOD2)
			for (i = 0; i < 3; i++) {
				if (direction == TR || direction == BL) {
					crd.column = 0;
					crd.row = 3 * i / 2;
					crd.diagonal = crd.row + i % 2;
				} else {
					crd.column = 3 * i / 2;
					crd.row = 3 * i / 2;
					crd.diagonal = crd.row + crd.column;
				}
				movePieces(w, face, crd, direction, style, True, fast);
			}
		else	/* (style == PERIOD3) */
			for (i = 0; i < 2; i++) {
				if (direction == TR || direction == BL) {
					crd.column = 1 + faceOnView;
					crd.row = 1 + faceOnView;
					crd.diagonal = crd.row + crd.column + i;
				} else {
					crd.column = i + 2 * faceOnView;
					crd.row = i + 2 * faceOnView;
					crd.diagonal = crd.row + crd.column + faceOnView;
				}
				movePieces(w, face, crd, direction, style, True, fast);
			}
	} else {
		for (i = 0; i < w->pyraminx.size; i++) {
			if (direction == TR || direction == BL) {
				if (style == PERIOD2) {
					crd.column = 0;
					crd.row = i;
					crd.diagonal = crd.row;
				} else {
					crd.column = faceOnView * (w->pyraminx.size - 1);
					crd.row = i;
					crd.diagonal = crd.column + crd.row + faceOnView;
				}
			} else {
				crd.column = i;
				crd.row = w->pyraminx.size - 1 - i;
				crd.diagonal = crd.column + crd.row + faceOnView;
			}
			movePieces(w, face, crd, direction, style, True, fast);
			/*setPuzzleMove(w, ACTION_MOVED, face, toPosition(w, crd),
				direction, style, 1, fast);*/
		}
	}
#ifdef HAVE_OPENGL
	/* if orient, then needs a redraw to get the lines right */
	/*if ((fast == INSTANT || w->pyraminx.orient) && w->pyraminx.dim == 4)*/
	if (fast == INSTANT && w->pyraminx.dim == 4) {
		drawAllPiecesGL((PyraminxGLWidget) w);
	}
#endif
}

#if 0
static Boolean
moveAllPieces(PyraminxWidget w, const int face, const int position,
		const int direction, const int style, const Boolean control,
		int fast)
{
	int size;
	Boolean newControl;
	CRD crd;

	size = w->pyraminx.size;
	toCRD(w, face, position, &crd);
	w->pyraminx.degreeTurn = 360 / style;
	if (control || size == 1) {
		newControl = True;
#if 0
		int k, newPosition, newFast;
		newFast = fast;
			newFast = INSTANT;
		for (k = 0; k < size; k++) {
			newPosition = k * k; /* FIXME */
			toCRD(w, face, newPosition, &crd);
			movePieces(w, face, crd, direction, style,
				newControl, newFast);
		}
#endif
	} else {
		newControl = False;
		movePieces(w, face, crd, direction, style, control, fast);
	}
	return newControl;
}
#endif

void
movePuzzle(PyraminxWidget w, int face, int position, int direction,
		int style, int control, int fast)
{
	if (direction == IGNORE_DIR) {
		return;
	}
	if (control) {
		moveControlCb(w, face, direction, style, fast);
		setPuzzleMove(w, ACTION_MOVED, face, 0, direction,
			style, True, fast);
	} else {
		CRD crd;

		toCRD(w, face, position, &crd);
		movePieces(w, face, crd, direction, style, False, fast);
		setPuzzleMove(w, ACTION_MOVED, face, position, direction,
			style, False, fast);
	}
#ifdef USE_SOUND
	if (w->pyraminx.sound) {
		playSound(MOVESOUND);
	}
#endif
	setMove(&undo, direction, style, control, face, position);
	flushMoves(w, &redo, FALSE);
}

void
movePuzzleDelay(PyraminxWidget w, int face, int position, int direction,
		int style, int control)
{
	movePuzzle(w, face, position, direction, style, control, NORMAL);
	Sleep((unsigned int) w->pyraminx.delay);
}

static Boolean
selectPieces(PyraminxWidget w, int positionX, int positionY,
		int *face, int *position)
{
	if (w->pyraminx.dim == 2)
		return selectPieces2D((Pyraminx2DWidget) w,
			positionX, positionY, face, position);
#ifdef HAVE_OPENGL
	else if (w->pyraminx.dim == 4)
		return selectPiecesGL((PyraminxGLWidget) w,
			positionX, positionY, face, position);
#endif
	return False;
}

static int
checkMoveDir(PyraminxWidget w, int face1, CRD crd1, int face2, CRD crd2,
		int style, int *direction)
{
	int which = -1, count = 0;
	int i, *p1, *p2;

	p1 = &(crd1.column);
	p2 = &(crd2.column);
	if (face1 == face2) {
		for (i = 0; i < 3; i++, p1++, p2++)
			if (*p1 == *p2) {
				which = i;
				count++;
			}
		if (count == 1)
			switch (which) {
			case 0:	/* COLUMN */
				*direction = (crd2.row > crd1.row) ?
					BOTTOM : TOP;
				break;
			case 1:	/* ROW */
				*direction = (crd2.column > crd1.column) ?
					RIGHT : LEFT;
				break;
			case 2:	/* DIAGONAL */
				*direction = (crd2.column > crd1.column) ?
					TR : BL;
				break;
			}
		if (!w->pyraminx.vertical && face1 >= MAX_SIDES &&
				*direction > LEFT)
			*direction = (*direction + MAX_SIDES) % MAX_FACES;
	} else {
		if (crd1.diagonal >= w->pyraminx.size)
			crd1.diagonal = 2 * w->pyraminx.size - 1 -
				crd1.diagonal;
		if (crd2.diagonal >= w->pyraminx.size)
			crd2.diagonal = 2 * w->pyraminx.size - 1 -
				crd2.diagonal;
		if (style == 2) {
			if (crd1.column == crd2.column) {
				which = 0;
				count++;
			}
			if (face1 / 2 == face2 / 2) {
				if (crd1.row == crd2.row) {
					which = 1;
					count++;
				}
				if (crd1.diagonal == crd2.diagonal) {
					which = 2;
					count++;
				}
			} else {
				if (crd1.row == w->pyraminx.size - 1 -
						crd2.row) {
					which = 1;
					count++;
				}
				if (crd1.diagonal == w->pyraminx.size - 1 -
						crd2.diagonal) {
					which = 2;
					count++;
				}
			}
		} else /* if (style == 3) */ {
			if (face1 == 0 && face2 == 1) {
				if (crd1.column == w->pyraminx.size - 1 -
						crd2.row) {
					which = 0;
					count++;
				}
				if (crd1.row == w->pyraminx.size - 1 -
						crd2.column) {
					which = 1;
					count++;
				}
			} else if (face1 == 0 && face2 == 2) {
				if (crd1.row == w->pyraminx.size - 1 -
						crd2.diagonal) {
					which = 1;
					count++;
				}
				if (crd1.diagonal == w->pyraminx.size - 1 -
						crd2.row) {
					which = 2;
					count++;
				}
			} else if (face1 == 0 && face2 == 3) {
				if (crd1.column == w->pyraminx.size - 1 -
						crd2.diagonal) {
					which = 0;
					count++;
				}
				if (crd1.diagonal == crd2.column) {
					which = 2;
					count++;
				}
			} else if (face1 == 1 && face2 == 0) {
				if (crd1.column == w->pyraminx.size - 1 -
						crd2.row) {
					which = 0;
					count++;
				}
				if (crd1.row == w->pyraminx.size - 1 -
						crd2.column) {
					which = 1;
					count++;
				}
			} else if (face1 == 1 && face2 == 2) {
				if (crd1.column == crd2.diagonal) {
					which = 0;
					count++;
				}
				if (crd1.diagonal == w->pyraminx.size - 1 -
						crd2.column) {
					which = 2;
					count++;
				}
			} else if (face1 == 1 && face2 == 3) {
				if (crd1.row == crd2.diagonal) {
					which = 1;
					count++;
				}
				if (crd1.diagonal == crd2.row) {
					which = 2;
					count++;
				}
			} else if (face1 == 2 && face2 == 0) {
				if (crd1.row == w->pyraminx.size - 1 -
						crd2.diagonal) {
					which = 1;
					count++;
				}
				if (crd1.diagonal == w->pyraminx.size - 1 -
						crd2.row) {
					which = 2;
					count++;
				}
			} else if (face1 == 2 && face2 == 1) {
				if (crd1.column == w->pyraminx.size - 1 -
						crd2.diagonal) {
					which = 0;
					count++;
				}
				if (crd1.diagonal == crd2.column) {
					which = 2;
					count++;
				}
			} else if (face1 == 2 && face2 == 3) {
				if (crd1.column == w->pyraminx.size - 1 -
						crd2.row) {
					which = 0;
					count++;
				}
				if (crd1.row == w->pyraminx.size - 1 -
						crd2.column) {
					which = 1;
					count++;
				}
			} else if (face1 == 3 && face2 == 0) {
				if (crd1.column == crd2.diagonal) {
					which = 0;
					count++;
				}
				if (crd1.diagonal == w->pyraminx.size - 1 -
						crd2.column) {
					which = 2;
					count++;
				}
			} else if (face1 == 3 && face2 == 1) {
				if (crd1.row == crd2.diagonal) {
					which = 1;
					count++;
				}
				if (crd1.diagonal == crd2.row) {
					which = 2;
					count++;
				}
			} else if (face1 == 3 && face2 == 2) {
				if (crd1.column == w->pyraminx.size - 1 -
						crd2.row) {
					which = 0;
					count++;
				}
				if (crd1.row == w->pyraminx.size - 1 -
						crd2.column) {
					which = 1;
					count++;
				}
			}
		}
		if (count == 1) {
			switch (which) {
			case 0:	/* COLUMN */
				if (directionNextFace[BOTTOM][face1] == face2) {
					*direction = BOTTOM;
				} else if (directionNextFace[TOP][face1] == face2) {
					*direction = TOP;
				} else {
					*direction = (NRAND(2) == 0) ? BOTTOM :
						TOP;
				}
				break;
			case 1: /* ROW */
				if (directionNextFace[LEFT][face1] == face2) {
					*direction = LEFT;
				} else if (directionNextFace[RIGHT][face1] == face2) {
					*direction = RIGHT;
				} else {
					*direction = (NRAND(2) == 0) ? LEFT :
						RIGHT;
				}
				break;
			case 2: /* DIAGONAL */
				if (directionNextFace[BL][face1] == face2) {
					*direction = BL;
				} else if (directionNextFace[TR][face1] == face2) {
					*direction = TR;
				} else {
					*direction = (NRAND(2) == 0) ? BL : TR;
				}
				break;
			default:
				count = 0;
			}
		}
		if (count >= 3)
			/* Very ambiguous but lets not ignore due to face change */
			count = 2;
	}
	return count;
}

static void
narrowSelection(PyraminxWidget w, int style,
		int *face, int *position, int *direction)
{
	if (w->pyraminx.dim == 2)
		narrowSelection2D((Pyraminx2DWidget) w, style,
			face, position, direction);
#ifdef HAVE_OPENGL
	else if (w->pyraminx.dim == 4)
		narrowSelectionGL((PyraminxGLWidget) w, style,
			face, position, direction);
#endif
}

static Boolean
positionPieces(PyraminxWidget w, int x, int y, int style,
		int *face, int *position, int *direction)
{
	if (!selectPieces(w, x, y, face, position))
		return False;
#if 0
	if (!w->pyraminx.vertical && *face >= MAX_SIDES &&
			*direction < MAX_ORIENT)
		*direction = (*direction + MAX_ORIENT / 2) % MAX_ORIENT;
#endif
	narrowSelection(w, style, face, position, direction);
	return True;
}

void
movePuzzleInput(PyraminxWidget w, int x, int y, int direction, int shift,
		int control)
{
	int style, face, position;

	if (w->pyraminx.mode == BOTH) {
		style = (shift) ? PERIOD3 : PERIOD2;
	} else {
		if (control && shift)
			style = (w->pyraminx.mode == PERIOD3) ?
				PERIOD2 : PERIOD3;
		else
			style = (w->pyraminx.mode == PERIOD2) ?
				PERIOD2 : PERIOD3;
	}
	if (!w->pyraminx.practice && !control && checkSolved(w)) {
		moveNoPieces(w);
		return;
	}
	if (!positionPieces(w, x, y, style, &face, &position, &direction))
		return;
	control = (control) ? 1 : 0;
	movePuzzle(w, face, position, direction,
		style, control, NORMAL);
	if (!control && checkSolved(w)) {
		setPuzzle(w, ACTION_SOLVED);
	}
}

void
resetPieces(PyraminxWidget w)
{
	int face, position, orient, side;

	w->pyraminx.sizeSize = w->pyraminx.size * w->pyraminx.size;
	for (face = 0; face < MAX_FACES; face++) {
		if (w->pyraminx.facetLoc[face])
			free(w->pyraminx.facetLoc[face]);
		if (!(w->pyraminx.facetLoc[face] = (PyraminxLoc *)
				malloc(sizeof (PyraminxLoc) *
					(size_t) w->pyraminx.sizeSize))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
		if (startLoc[face])
			free(startLoc[face]);
		if (!(startLoc[face] = (PyraminxLoc *)
				malloc(sizeof (PyraminxLoc) *
					(size_t) w->pyraminx.sizeSize))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
	for (orient = 0; orient < 3; orient++)
		for (side = 0; side < MAX_SIDES; side++) {
			if (w->pyraminx.rowLoc[orient][side])
				free(w->pyraminx.rowLoc[orient][side]);
			if (!(w->pyraminx.rowLoc[orient][side] = (PyraminxLoc *)
					malloc(sizeof (PyraminxLoc) *
						(size_t) w->pyraminx.size))) {
				DISPLAY_ERROR("Not enough memory, exiting.");
			}
		}
	for (side = 0; side < MAX_SIDES; side++) {
		if (w->pyraminx.faceLoc[side])
			free(w->pyraminx.faceLoc[side]);
		if (!(w->pyraminx.faceLoc[side] = (PyraminxLoc *)
				malloc(sizeof (PyraminxLoc) *
					(size_t) w->pyraminx.sizeSize))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
	for (face = 0; face < MAX_FACES; face++)
		for (position = 0; position < w->pyraminx.sizeSize; position++) {
			w->pyraminx.facetLoc[face][position].face = face;
			w->pyraminx.facetLoc[face][position].rotation = TOP;
		}
	flushMoves(w, &undo, TRUE);
	flushMoves(w, &redo, FALSE);
	w->pyraminx.started = False;
	w->pyraminx.currentFace = IGNORE_DIR;
	w->pyraminx.currentDirection = IGNORE_DIR;
	w->pyraminx.currentStyle = IGNORE_DIR;
}

static void
readFile(PyraminxWidget w, FILE *fp, char *name)
{
	int c, i, size, mode, sticky, orient, practice, moves;
	char *buf1 = NULL, *buf2 = NULL;

	while ((c = getc(fp)) != EOF && c != SYMBOL);
	if (fscanf(fp, "%d", &size) != 1) {
		(void) printf("corrupt record, expecting an integer for size\n");
		return;
	}
	if (size >= MIN_FACETS) {
		for (i = w->pyraminx.size; i < size; i++) {
			setPuzzle(w, ACTION_INCREMENT);
		}
		for (i = w->pyraminx.size; i > size; i--) {
			setPuzzle(w, ACTION_DECREMENT);
		}
	} else {
		stringCat(&buf1, name, " corrupted: size ");
		intCat(&buf2, buf1, size);
		free(buf1);
		stringCat(&buf1, buf2, " should be between ");
		free(buf2);
		intCat(&buf2, buf1, MIN_FACETS);
		free(buf1);
		stringCat(&buf1, buf2, " and MAXINT");
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		return;
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	if (fscanf(fp, "%d", &mode) != 1) {
		(void) printf("corrupt record, expecting an integer for mode\n");
		return;
	}
	switch (mode) {
	case PERIOD2:
		setPuzzle(w, ACTION_PERIOD2);
		break;
	case PERIOD3:
		setPuzzle(w, ACTION_PERIOD3);
		break;
	case BOTH:
		setPuzzle(w, ACTION_BOTH);
		break;
	default:
		stringCat(&buf1, name, " corrupted: mode ");
		intCat(&buf2, buf1, mode);
		free(buf1);
		stringCat(&buf1, buf2, " should be between ");
		free(buf2);
		intCat(&buf2, buf1, PERIOD2);
		free(buf1);
		stringCat(&buf1, buf2, " and ");
		free(buf2);
		intCat(&buf2, buf1, BOTH);
		free(buf1);
		DISPLAY_WARNING(buf2);
		free(buf2);
		return;
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	if (fscanf(fp, "%d", &sticky) != 1) {
		(void) printf("corrupt record, expecting an integer for sticky\n");
		return;
	}
	if (w->pyraminx.sticky != (Boolean) sticky) {
		setPuzzle(w, ACTION_STICKY);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	if (fscanf(fp, "%d", &orient) != 1) {
		(void) printf("corrupt record, expecting an integer for orient\n");
		return;
	}
	if (w->pyraminx.orient != (Boolean) orient) {
		setPuzzle(w, ACTION_ORIENTIZE);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	if (fscanf(fp, "%d", &practice) != 1) {
		(void) printf("corrupt record, expecting an integer for practice\n");
		return;
	}
	if (w->pyraminx.practice != (Boolean) practice) {
		setPuzzle(w, ACTION_PRACTICE);
	}
#ifdef WINVER
	resetPieces(w);
#endif
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	if (fscanf(fp, "%d", &moves) != 1) {
		(void) printf("corrupt record, expecting an integer for moves\n");
		return;
	}
	if (!scanStartPosition(fp, w))
		return;
	setPuzzle(w, ACTION_RESTORE);
	setStartPosition(w);
	if (!scanMoves(fp, w, moves))
		return;
	(void) printf("%s: size %d, mode %d, sticky %s, orient %s",
		name, size, mode, BOOL_STRING(sticky), BOOL_STRING(orient));
	(void) printf(", practice %s, moves %d\n",
		BOOL_STRING(practice), moves);
}

static void
getPieces(PyraminxWidget w)
{
	FILE *fp;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "r")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Cannot read ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Cannot read ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			return;
		}
#endif
	}
	flushMoves(w, &undo, TRUE);
	flushMoves(w, &redo, FALSE);
	readFile(w, fp, name);
	(void) fclose(fp);
	free(lname);
	free(fname);
	w->pyraminx.cheat = True; /* Assume the worst. */
}

static void
writePieces(PyraminxWidget w)
{
	FILE *fp;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "w")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "w")) == NULL) {
			stringCat(&buf1, "Cannot write to ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Cannot write to ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			return;
		}
#endif
	}
	(void) fprintf(fp, "size%c %d\n", SYMBOL, w->pyraminx.size);
	(void) fprintf(fp, "mode%c %d\n", SYMBOL, w->pyraminx.mode);
	(void) fprintf(fp, "sticky%c %d\n", SYMBOL,
		(w->pyraminx.sticky) ? 1 : 0);
	(void) fprintf(fp, "orient%c %d\n", SYMBOL,
		(w->pyraminx.orient) ? 1 : 0);
	(void) fprintf(fp, "practice%c %d\n", SYMBOL,
		(w->pyraminx.practice) ? 1 : 0);
	(void) fprintf(fp, "moves%c %d\n", SYMBOL,
		numMoves(&undo));
	printStartPosition(fp, w);
	printMoves(fp, &undo);
	(void) fclose(fp);
	(void) printf("Saved to %s\n", name);
	free(lname);
	free(fname);
}

static void
undoPieces(PyraminxWidget w)
{
	if (madeMoves(&undo) &&
			w->pyraminx.currentFace <= IGNORE_DIR) {
		int face, position, direction, style, control;

		getMove(&undo, &direction, &style, &control,
			&face, &position);
		setMove(&redo, direction, style, control,
			face, position);
		direction = (direction < MAX_ORIENT) ? (direction + MAX_ORIENT / 2) %
			MAX_ORIENT : 3 * MAX_ORIENT - direction;
#if 0
		setPuzzleMove(w, ACTION_UNDO, face, position, direction, style,
			moveAllPieces(w, face, position, direction, style,
			control, DOUBLE), DOUBLE);
		if (!control && checkSolved(w)) {
			setPuzzle(w, ACTION_SOLVED);
		}
#else
		if (control) {
			moveControlCb(w, face, direction, style, DOUBLE);
			setPuzzleMove(w, ACTION_MOVED, face, 0, direction,
				style, True, DOUBLE);
		} else {
			CRD crd;

			toCRD(w, face, position, &crd);
			movePieces(w, face, crd, direction, style, False, DOUBLE);
			setPuzzleMove(w, ACTION_UNDO, face, position, direction,
				style, False, DOUBLE);
			if (checkSolved(w)) {
				setPuzzle(w, ACTION_SOLVED);
			}
		}
#endif
	}
}

static void
redoPieces(PyraminxWidget w)
{
	if (madeMoves(&redo) &&
			w->pyraminx.currentFace <= IGNORE_DIR) {
		int face, position, direction, style, control;

		getMove(&redo, &direction, &style, &control,
			&face, &position);
		setMove(&undo, direction, style, control,
			face, position);
#if 0
		setPuzzleMove(w, ACTION_REDO, face, position, direction, style,
			moveAllPieces(w, face, position, direction, style,
			control, DOUBLE), DOUBLE);
		if (!control && checkSolved(w)) {
			setPuzzle(w, ACTION_SOLVED);
		}
#else
		if (control) {
			moveControlCb(w, face, direction, style, DOUBLE);
			setPuzzleMove(w, ACTION_MOVED, face, 0, direction,
				style, True, DOUBLE);
		} else {
			CRD crd;

			toCRD(w, face, position, &crd);
			movePieces(w, face, crd, direction, style, False, DOUBLE);
			setPuzzleMove(w, ACTION_REDO, face, position, direction,
				style, False, DOUBLE);
			if (checkSolved(w)) {
				setPuzzle(w, ACTION_SOLVED);
			}
		}
#endif
	}
}

void
clearPieces(PyraminxWidget w)
{
	setPuzzle(w, ACTION_CLEAR);
}

static void
practicePieces(PyraminxWidget w)
{
	setPuzzle(w, ACTION_PRACTICE);
}

static void
randomizePieces(PyraminxWidget w)
{
	int randomDirection, face, position, style;
	int big = w->pyraminx.sizeSize * 3 + NRAND(2);

	if (w->pyraminx.currentFace > IGNORE_DIR)
		return;
	w->pyraminx.cheat = False;
	if (w->pyraminx.practice)
		practicePieces(w);
	setPuzzle(w, ACTION_RESET);
	if (big > 1000)
		big = 1000;
	if (w->pyraminx.sticky)
		big /= 3;

#ifdef DEBUG
	big = 3;
#endif

	while (big--) {
		face = NRAND(MAX_FACES);
		if (w->pyraminx.mode == BOTH)
			style = NRAND(MAX_MODES - 1) + PERIOD2;
		else
			style = w->pyraminx.mode;
		if (w->pyraminx.sticky) {
			if (style == PERIOD2) {
				if (NRAND(3) == 2) {
					position = (NRAND(2)) ? 9 : 6;
					randomDirection = (NRAND(2)) ? TR : BL;
				} else {
					position = (NRAND(2)) ? 6 : 0;
					if (NRAND(2))
						randomDirection = (NRAND(2)) ? LEFT : RIGHT;
					else
						randomDirection = (NRAND(2)) ? TOP : BOTTOM;
				}
			} else {	/* style == PERIOD3 */
				position = 6;
				randomDirection = NRAND(6);
			}
		} else {	/* (!w->pyraminx.sticky) */
			randomDirection = NRAND(MAX_ORIENT);
			position = NRAND(w->pyraminx.sizeSize);
			if (w->pyraminx.mode == BOTH)
				style = NRAND(MAX_MODES - 1) + PERIOD2;
			else
				style = w->pyraminx.mode;
		}
		movePuzzle(w, face, position, randomDirection,
			style, FALSE, INSTANT);
		setPuzzle(w, ACTION_MOVED);
	}
	flushMoves(w, &undo, TRUE);
	flushMoves(w, &redo, FALSE);
	setPuzzle(w, ACTION_RANDOMIZE);
	if (checkSolved(w)) {
		setPuzzle(w, ACTION_SOLVED);
	}
}

static void
solvePieces(PyraminxWidget w)
{
#ifdef TRY_SOLVE
	solveSomePieces(w);
#else
	if (checkSolved(w) || w->pyraminx.currentFace > IGNORE_DIR)
		return;
	if (((w->pyraminx.size <= 3 || w->pyraminx.sticky) && w->pyraminx.mode == PERIOD2) ||
			(w->pyraminx.size <= 4 && w->pyraminx.mode == PERIOD3))
		solveSomePieces(w);
	else {
		setPuzzle(w, ACTION_SOLVE_MESSAGE);
	}
#endif
}

static void
findPieces(PyraminxWidget w)
{
#ifdef FIND
	if ((w->pyraminx.size == 4 && w->pyraminx.mode == PERIOD3) ||
			(w->pyraminx.size == 3 && w->pyraminx.mode == PERIOD2))
		findSomeMoves(w);
	else {
		setPuzzle(w, ACTION_FIND_MESSAGE);
	}
#endif
}

static void
incrementPieces(PyraminxWidget w)
{
	setPuzzle(w, ACTION_INCREMENT);
}

static Boolean
decrementPieces(PyraminxWidget w)
{
	if (w->pyraminx.size <= MIN_FACETS)
		return False;
	setPuzzle(w, ACTION_DECREMENT);
	return True;
}

static void
orientizePieces(PyraminxWidget w)
{
	setPuzzle(w, ACTION_ORIENTIZE);
}

static void
stickyPieces(PyraminxWidget w)
{
	setPuzzle(w, ACTION_STICKY);
}

static void
viewPieces(PyraminxWidget w)
{
	setPuzzle(w, ACTION_VIEW);
}

static void
speedPieces(PyraminxWidget w)
{
	w->pyraminx.delay -= 5;
	if (w->pyraminx.delay < 0)
		w->pyraminx.delay = 0;
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	setPuzzle(w, ACTION_SPEED);
#endif
}

static void
slowPieces(PyraminxWidget w)
{
	w->pyraminx.delay += 5;
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	setPuzzle(w, ACTION_SPEED);
#endif
}

static void
soundPieces(PyraminxWidget w)
{
	w->pyraminx.sound = !w->pyraminx.sound;
	setPuzzle(w, ACTION_SOUND);
}

#ifdef WINVER
static void
setValuesPuzzle(PyraminxWidget w)
{
	struct tagColor {
		int red, green, blue;
	} color;
	char szBuf[80], buf[20], charbuf[2];
	int face;

	w->pyraminx.size = (int) GetPrivateProfileInt(SECTION,
		"size", DEFAULT_FACETS, INIFILE);
	w->pyraminx.mode = (int) GetPrivateProfileInt(SECTION,
		"mode", DEFAULT_MODE, INIFILE);
	w->pyraminx.sticky = (BOOL) GetPrivateProfileInt(SECTION,
		"sticky", DEFAULT_STICKY, INIFILE);
	w->pyraminx.orient = (BOOL) GetPrivateProfileInt(SECTION,
		"orient", DEFAULT_ORIENT, INIFILE);
	w->pyraminx.practice = (BOOL) GetPrivateProfileInt(SECTION,
		"practice", DEFAULT_PRACTICE, INIFILE);
/*#ifdef HAVE_OPENGL
	w->pyraminx.dim = (int) GetPrivateProfileInt(SECTION,
		"dim", 4, INIFILE);
#else
#endif FIXME when GL select working */
	w->pyraminx.dim = (int) GetPrivateProfileInt(SECTION,
		"dim", 2, INIFILE);
	w->pyraminx.view = (int) GetPrivateProfileInt(SECTION,
		"view", 1, INIFILE);
	w->pyraminx.mono = (BOOL) GetPrivateProfileInt(SECTION,
		"mono", DEFAULT_MONO, INIFILE);
	w->pyraminx.reverse = (BOOL) GetPrivateProfileInt(SECTION,
		"reverseVideo", DEFAULT_REVERSE, INIFILE);
	/* cyan */
	(void) GetPrivateProfileString(SECTION,
		"frameColor", "0 255 255", szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->pyraminx.frameGC = RGB(color.red, color.green, color.blue);
	/* gray25 */
	(void) GetPrivateProfileString(SECTION,
		"pieceBorder", "64 64 64", szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->pyraminx.borderGC = RGB(color.red, color.green, color.blue);
	/* #AEB2C3 */
	(void) GetPrivateProfileString(SECTION,
		"background", "174 178 195", szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->pyraminx.inverseGC = RGB(color.red, color.green, color.blue);
	for (face = 0; face < MAX_FACES; face++) {
#ifdef HAVE_SNPRINTF
		(void) snprintf(buf, 20, "faceColor%d", face);
#else
		(void) sprintf(buf, "faceColor%d", face);
#endif
		(void) GetPrivateProfileString(SECTION,
			buf, faceColorString[face],
			szBuf, sizeof (szBuf), INIFILE);
		(void) sscanf(szBuf, "%d %d %d",
			&(color.red), &(color.green), &(color.blue));
		w->pyraminx.faceGC[face] =
			RGB(color.red, color.green, color.blue);
#ifdef HAVE_SNPRINTF
		(void) snprintf(buf, 20, "faceChar%d", face);
#else
		(void) sprintf(buf, "faceChar%d", face);
#endif
		charbuf[0] = faceColorChar[face];
		charbuf[1] = '\0';
		(void) GetPrivateProfileString(SECTION,
			buf, charbuf, szBuf, sizeof (szBuf), INIFILE);
		w->pyraminx.faceChar[face] = szBuf[0];
	}
	w->pyraminx.delay = (int) GetPrivateProfileInt(SECTION,
		"delay", 10, INIFILE);
	w->pyraminx.sound = (BOOL) GetPrivateProfileInt(SECTION,
		"sound", FALSE, INIFILE);
	(void) GetPrivateProfileString(SECTION,
		"moveSound", MOVESOUND, szBuf, sizeof (szBuf), INIFILE);
	(void) strncpy(w->pyraminx.moveSound, szBuf, 80);
	(void) GetPrivateProfileString(SECTION,
		"userName", "guest", szBuf, sizeof (szBuf), INIFILE);
	(void) strncpy(w->pyraminx.userName, szBuf, 80);
	(void) GetPrivateProfileString(SECTION,
		"scoreFile", "", szBuf, sizeof (szBuf), INIFILE);
	(void) strncpy(w->pyraminx.scoreFile, szBuf, 80);
}

void
destroyPuzzle(HBRUSH brush)
{
	deleteMoves(&undo);
	deleteMoves(&redo);
	(void) DeleteObject(brush);
	PostQuitMessage(0);
}

void
resizePuzzle(PyraminxWidget w)
{
	if (w->pyraminx.dim == 2)
		resizePuzzle2D((Pyraminx2DWidget) w);
#ifdef HAVE_OPENGL
	else if (w->pyraminx.dim == 4)
		resizePuzzleGL((PyraminxGLWidget) w);
#endif
}

void
sizePuzzle(PyraminxWidget w)
{
	resetPieces(w);
	resizePuzzle(w);
}

void
exposePuzzle(PyraminxWidget w)
{
	if (w->pyraminx.dim == 2)
		exposePuzzle2D((Pyraminx2DWidget) w);
#ifdef HAVE_OPENGL
	else if (w->pyraminx.dim == 4)
		exposePuzzleGL((PyraminxGLWidget) w);
#endif
}

#else
static void
getColor(PyraminxWidget w, int face)
{
	XGCValues values;
	XtGCMask valueMask;
	XColor colorCell, rgb;

	valueMask = GCForeground | GCBackground;
	if (w->pyraminx.reverse) {
		values.background = w->pyraminx.foreground;
	} else {
		values.background = w->pyraminx.background;
	}
	if (!w->pyraminx.mono || w->pyraminx.dim == 4) {
		if (XAllocNamedColor(XtDisplay(w),
				DefaultColormapOfScreen(XtScreen(w)),
				w->pyraminx.faceName[face], &colorCell, &rgb)) {
			values.foreground = w->pyraminx.faceColor[face] = colorCell.pixel;
			if (w->pyraminx.faceGC[face])
				XtReleaseGC((Widget) w,
					w->pyraminx.faceGC[face]);
			w->pyraminx.faceGC[face] = XtGetGC((Widget) w, valueMask, &values);
			return;
		} else {
			char *buf1, *buf2;

			stringCat(&buf1, "Color name \"",
				w->pyraminx.faceName[face]);
			stringCat(&buf2, buf1, "\" is not defined for face ");
			free(buf1);
			intCat(&buf1, buf2, face);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
	}
	if (w->pyraminx.reverse) {
		values.background = w->pyraminx.foreground;
		values.foreground = w->pyraminx.background;
	} else {
		values.background = w->pyraminx.background;
		values.foreground = w->pyraminx.foreground;
	}
	if (w->pyraminx.faceGC[face])
		XtReleaseGC((Widget) w, w->pyraminx.faceGC[face]);
	w->pyraminx.faceGC[face] = XtGetGC((Widget) w, valueMask, &values);
}

void
setAllColors(PyraminxWidget w)
{
	XGCValues values;
	XtGCMask valueMask;
	int face;

	valueMask = GCForeground | GCBackground;

	if (w->pyraminx.reverse) {
		values.background = w->pyraminx.background;
		values.foreground = w->pyraminx.foreground;
	} else {
		values.foreground = w->pyraminx.background;
		values.background = w->pyraminx.foreground;
	}
	if (w->pyraminx.inverseGC)
		XtReleaseGC((Widget) w, w->pyraminx.inverseGC);
	w->pyraminx.inverseGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->pyraminx.mono) {
		if (w->pyraminx.reverse) {
			values.background = w->pyraminx.foreground;
			values.foreground = w->pyraminx.background;
		} else {
			values.foreground = w->pyraminx.foreground;
			values.background = w->pyraminx.background;
		}
	} else {
		values.foreground = w->pyraminx.frameColor;
		values.background = w->pyraminx.background;
	}
	if (w->pyraminx.frameGC)
		XtReleaseGC((Widget) w, w->pyraminx.frameGC);
	w->pyraminx.frameGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->pyraminx.mono) {
		if (w->pyraminx.reverse) {
			values.background = w->pyraminx.foreground;
			values.foreground = w->pyraminx.background;
		} else {
			values.foreground = w->pyraminx.foreground;
			values.background = w->pyraminx.background;
		}
	} else {
		values.foreground = w->pyraminx.borderColor;
		values.background = w->pyraminx.background;
	}
	if (w->pyraminx.borderGC)
		XtReleaseGC((Widget) w, w->pyraminx.borderGC);
	w->pyraminx.borderGC = XtGetGC((Widget) w, valueMask, &values);
	for (face = 0; face < MAX_FACES; face++)
		getColor(w, face);
	if (w->pyraminx.fontInfo)
		XSetFont(XtDisplay(w), w->pyraminx.borderGC,
			w->pyraminx.fontInfo->fid);
}

static Boolean
setValuesPuzzle(Widget current, Widget request, Widget renew)
{
	PyraminxWidget c = (PyraminxWidget) current, w = (PyraminxWidget) renew;
	Boolean redraw = False, setColors = False;
	int face;

	checkPieces(w);
	for (face = 0; face < MAX_FACES; face++) {
		if (strcmp(w->pyraminx.faceName[face], c->pyraminx.faceName[face])) {
			setColors = True;
			break;
		}
	}
	if (w->pyraminx.font != c->pyraminx.font ||
			w->pyraminx.borderColor != c->pyraminx.borderColor ||
			w->pyraminx.reverse != c->pyraminx.reverse ||
			w->pyraminx.mono != c->pyraminx.mono) {
		loadFont(w);
		setAllColors(w);
		redraw = True;
	} else if (w->pyraminx.background != c->pyraminx.background ||
			w->pyraminx.foreground != c->pyraminx.foreground ||
			setColors) {
		setAllColors(w);
		redraw = True;
	}
	if (w->pyraminx.orient != c->pyraminx.orient ||
			w->pyraminx.sticky != c->pyraminx.sticky ||
			w->pyraminx.mode != c->pyraminx.mode ||
			w->pyraminx.practice != c->pyraminx.practice) {
		resetPieces(w);
		redraw = True;
	}
	if (w->pyraminx.menu != ACTION_IGNORE) {
		int menu = w->pyraminx.menu;

		w->pyraminx.menu = ACTION_IGNORE;
		switch (menu) {
		case ACTION_GET:
			getPieces(w);
			break;
		case ACTION_WRITE:
			writePieces(w);
			break;
		case ACTION_UNDO:
			undoPieces(w);
			break;
		case ACTION_REDO:
			redoPieces(w);
			break;
		case ACTION_CLEAR:
			clearPieces(w);
			break;
		case ACTION_PRACTICE:
			practicePieces(w);
			break;
		case ACTION_RANDOMIZE:
			randomizePieces(w);
			break;
		case ACTION_SOLVE:
			solvePieces(w);
			break;
		case ACTION_FIND:
			findPieces(w);
			break;
		case ACTION_INCREMENT:
			incrementPieces(w);
			break;
		case ACTION_DECREMENT:
			(void) decrementPieces(w);
			break;
		case ACTION_ORIENTIZE:
			orientizePieces(w);
			break;
		case ACTION_STICKY:
			stickyPieces(w);
			break;
		case ACTION_SPEED:
			speedPieces(w);
			break;
		case ACTION_SLOW:
			slowPieces(w);
			break;
		case ACTION_SOUND:
			soundPieces(w);
			break;
		case ACTION_VIEW:
			viewPieces(w);
			break;
		default:
			break;
		}
	}
	if (w->pyraminx.currentDirection == RESTORE_DIR) {
		setStartPosition(w);
		w->pyraminx.currentDirection = IGNORE_DIR;
		w->pyraminx.currentStyle = IGNORE_DIR;
	} else if (w->pyraminx.currentDirection == CLEAR_DIR) {
		w->pyraminx.currentDirection = IGNORE_DIR;
		w->pyraminx.currentStyle = IGNORE_DIR;
		resetPieces(w);
		redraw = True;
		w->pyraminx.currentDirection = IGNORE_DIR;
	} else if (w->pyraminx.currentDirection > IGNORE_DIR) {
		int face = w->pyraminx.currentFace;
		int position = w->pyraminx.currentPosition;
		int direction = w->pyraminx.currentDirection;
		int style = w->pyraminx.currentStyle;

		w->pyraminx.currentFace = IGNORE_DIR;
		w->pyraminx.currentPosition = IGNORE_DIR;
		w->pyraminx.currentDirection = IGNORE_DIR;
		w->pyraminx.currentStyle = IGNORE_DIR;
		if (w->pyraminx.currentControl) {
			moveControlCb(w, face, direction, style,
				w->pyraminx.currentFast);
		} else {
			CRD crd;
			toCRD(w, face, position, &crd);
			(void) movePieces(w, face, crd, direction, style,
				w->pyraminx.currentControl, w->pyraminx.currentFast);
		}
	}
	return (redraw);
}

static void
destroyPuzzle(Widget old)
{
	PyraminxWidget w = (PyraminxWidget) old;
	Display *display = XtDisplay(w);
	int face;

#if defined( USE_SOUND ) && defined( USE_ESOUND )
	if (w->pyraminx.dim == 2)
		(void) shutdown_sound();
#endif
	for (face = 0; face < MAX_FACES; face++)
		XtReleaseGC(old, w->pyraminx.faceGC[face]);
	XtReleaseGC(old, w->pyraminx.borderGC);
	XtReleaseGC(old, w->pyraminx.frameGC);
	XtReleaseGC(old, w->pyraminx.inverseGC);
	XtRemoveCallbacks(old, XtNselectCallback, w->pyraminx.select);
	if (w->pyraminx.fontInfo) {
		XUnloadFont(display, w->pyraminx.fontInfo->fid);
		XFreeFont(display, w->pyraminx.fontInfo);
	}
	deleteMoves(&undo);
	deleteMoves(&redo);
}

void
quitPuzzle(PyraminxWidget w, XEvent *event, char **args, int nArgs)
{
	XtCloseDisplay(XtDisplay(w));
	exit(0);
}
#endif

#ifndef WINVER
static
#endif
void
initializePuzzle(
#ifdef WINVER
PyraminxWidget w, HBRUSH brush
#else
Widget request, Widget renew
#endif
)
{
	int face, orient, side;
#ifdef WINVER
	setValuesPuzzle(w);
#else
	PyraminxWidget w = (PyraminxWidget) renew;

	w->pyraminx.mono = (DefaultDepthOfScreen(XtScreen(w)) < 2 ||
		w->pyraminx.mono);
	w->pyraminx.fontInfo = NULL;
	for (face = 0; face < MAX_FACES; face++)
		w->pyraminx.faceGC[face] = NULL;
	w->pyraminx.borderGC = NULL;
	w->pyraminx.frameGC = NULL;
	w->pyraminx.inverseGC = NULL;
#endif
	w->pyraminx.focus = False;
	loadFont(w);
	for (face = 0; face < MAX_FACES; face++)
		w->pyraminx.facetLoc[face] = NULL;
	for (orient = 0; orient < 3; orient++)
		for (side = 0; side < MAX_SIDES; side++)
			w->pyraminx.rowLoc[orient][side] = NULL;
	for (side = 0; side < MAX_SIDES; side++)
		w->pyraminx.faceLoc[side] = NULL;
	checkPieces(w);
	newMoves(&undo);
	newMoves(&redo);
	w->pyraminx.cheat = False;
	resetPieces(w);
#ifdef WINVER
	brush = CreateSolidBrush(w->pyraminx.inverseGC);
	SETBACK(w->core.hWnd, brush);
	(void) SRAND(time(NULL));
#else
	(void) SRAND(getpid());
	setAllColors(w);
#endif
#ifdef USE_SOUND
#ifdef USE_NAS
	dsp = XtDisplay(w);
#endif
#ifdef USE_ESOUND
	if (w->pyraminx.dim == 2)
		(void) init_sound();
#endif
#endif
}

void
hidePuzzle(PyraminxWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	setPuzzle(w, ACTION_HIDE);
}

void
selectPuzzle(PyraminxWidget w
#ifdef WINVER
, const int x, const int y, const int control
#else
, XEvent *event, char **args, int nArgs
#endif
)
{
#ifndef WINVER
	int x = event->xbutton.x, y = event->xbutton.y;
	int control = (int) (event->xkey.state & ControlMask);
#endif

	if (selectPieces(w, x, y, &(w->pyraminx.currentFace), &(w->pyraminx.currentPosition))) {
		if (w->pyraminx.currentPosition >= w->pyraminx.sizeSize) {
			w->pyraminx.currentFace = IGNORE_DIR;
			return;
		}
		if (control || w->pyraminx.practice || !checkSolved(w))
			drawTriangleAll(w, w->pyraminx.currentFace,
				w->pyraminx.currentPosition, TRUE);
	} else {
		w->pyraminx.currentFace = IGNORE_DIR;
		w->pyraminx.currentDirection = IGNORE_DIR;
#ifdef HAVE_OPENGL
		if (w->pyraminx.dim == 4)
			drawAllPiecesGL((PyraminxGLWidget) w);
#endif
	}
}

void
releasePuzzle(PyraminxWidget w
#ifdef WINVER
, const int x, const int y, const int shift, const int control
#else
, XEvent *event, char **args, int nArgs
#endif
)
{
	int style, face, position, count = -1, direction = 0;
#ifndef WINVER
	int x = event->xbutton.x, y = event->xbutton.y;
	int shift = (int) (event->xkey.state & (ShiftMask | LockMask));
	int control = (int) (event->xkey.state & ControlMask);
#endif


	if (w->pyraminx.currentFace <= IGNORE_DIR)
		return;
	drawTriangle(w, w->pyraminx.currentFace, w->pyraminx.currentPosition,
		FALSE);
	if (!control && !w->pyraminx.practice && checkSolved(w))
		moveNoPieces(w);
	else if (selectPieces(w, x, y, &face, &position)) {
		CRD currentCrd, crd;

		if (w->pyraminx.mode == BOTH) {
			style = (shift) ? PERIOD3 : PERIOD2;
		} else {
			if (control && shift)
				style = (w->pyraminx.mode == PERIOD3) ? PERIOD2 : PERIOD3;
			else
				style = (w->pyraminx.mode == PERIOD2) ? PERIOD2 : PERIOD3;
		}
		toCRD(w, w->pyraminx.currentFace, w->pyraminx.currentPosition,
			&currentCrd);
		toCRD(w, face, position, &crd);
		count = checkMoveDir(w, w->pyraminx.currentFace, currentCrd,
			face, crd, style, &direction);
		if (count == 1) {
			movePuzzle(w, w->pyraminx.currentFace,
				w->pyraminx.currentPosition,
				direction, style, (control) ? 1 : 0, NORMAL);
			if (!control && checkSolved(w)) {
				setPuzzle(w, ACTION_SOLVED);
			}
		} else if (count == 2) {
			/* count == 3 too annoying */
			setPuzzle(w, ACTION_AMBIGUOUS);
		} else if (count == 0)
			moveNoPieces(w);
	}
	w->pyraminx.currentFace = IGNORE_DIR;
	w->pyraminx.currentDirection = IGNORE_DIR;
#ifdef HAVE_OPENGL
	if (w->pyraminx.dim == 4 && count != 1) {
		drawAllPiecesGL((PyraminxGLWidget) w);
	}
#endif
}

#ifndef WINVER
void
practicePuzzleWithQuery(PyraminxWidget w
, XEvent *event, char **args, int nArgs
)
{
	if (!w->pyraminx.started)
		practicePieces(w);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	else {
		setPuzzle(w, ACTION_PRACTICE_QUERY);
	}
#endif
}

void
practicePuzzleWithDoubleClick(PyraminxWidget w
, XEvent *event, char **args, int nArgs
)
{
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	if (!w->pyraminx.started)
#endif
		practicePieces(w);
}
#endif

#ifndef WINVER
void
randomizePuzzleWithQuery(PyraminxWidget w
, XEvent *event, char **args, int nArgs
)
{
	if (!w->pyraminx.started)
		randomizePieces(w);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	else {
		setPuzzle(w, ACTION_RANDOMIZE_QUERY);
	}
#endif
}

void
randomizePuzzleWithDoubleClick(PyraminxWidget w
, XEvent *event, char **args, int nArgs
)
{
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	if (!w->pyraminx.started)
#endif
		randomizePieces(w);
}
#endif

void
getPuzzle(PyraminxWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	getPieces(w);
}

void
writePuzzle(PyraminxWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	writePieces(w);
}

void
undoPuzzle(PyraminxWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	undoPieces(w);
}

void
redoPuzzle(PyraminxWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	redoPieces(w);
}

void
clearPuzzle(PyraminxWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	clearPieces(w);
}

void
randomizePuzzle(PyraminxWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	randomizePieces(w);
}

void
solvePuzzle(PyraminxWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	solvePieces(w);
}

void
findPuzzle(PyraminxWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	findPieces(w);
}

void
practicePuzzle(PyraminxWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	practicePieces(w);
}

void
incrementPuzzle(PyraminxWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	incrementPieces(w);
}

#ifdef WINVER
Boolean
#else
void
#endif
decrementPuzzle(PyraminxWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
#ifdef WINVER
	return
#else
	(void)
#endif
	decrementPieces(w);
}

void
orientizePuzzle(PyraminxWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	orientizePieces(w);
}

void
stickyModePuzzle(PyraminxWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	stickyPieces(w);
}

#ifndef WINVER
void
period2ModePuzzle(PyraminxWidget w
, XEvent *event, char **args, int nArgs
)
{
	setPuzzle(w, ACTION_PERIOD2);
}

void
period3ModePuzzle(PyraminxWidget w
, XEvent *event, char **args, int nArgs
)
{
	setPuzzle(w, ACTION_PERIOD3);
}

void
bothModePuzzle(PyraminxWidget w
, XEvent *event, char **args, int nArgs
)
{
	setPuzzle(w, ACTION_BOTH);
}
#endif

void
viewPuzzle(PyraminxWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	viewPieces(w);
}

void
speedUpPuzzle(PyraminxWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	speedPieces(w);
}

void
slowDownPuzzle(PyraminxWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	slowPieces(w);
}

void
toggleSoundPuzzle(PyraminxWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	soundPieces(w);
}

void
enterPuzzle(PyraminxWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	w->pyraminx.focus = True;
	drawFrame(w, w->pyraminx.focus);
}

void
leavePuzzle(PyraminxWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	w->pyraminx.focus = False;
	drawFrame(w, w->pyraminx.focus);
}

#ifdef WINVER
void
periodModePuzzle(PyraminxWidget w, const int mode)
{
	setPuzzle(w, mode + ACTION_PERIOD2);
}

void
dimPuzzle(PyraminxWidget w)
{
	setPuzzle(w, ACTION_DIM);
}

#else

void
movePuzzleCcw(PyraminxWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput(w, event->xbutton.x, event->xbutton.y, CCW,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}

void
movePuzzleCw(PyraminxWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput(w, event->xbutton.x, event->xbutton.y, CW,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}
#endif
